
## ----- Shared -------------------------------------------------------------------

#version 140		// Needed for isamplerBuffer



## ----- Vertex -------------------------------------------------------------------

in vec4 position;
out vec3 LocalOffset;

uniform ivec4 ActiveRect;
uniform ivec2 GameResolution;
uniform int WaterLevel;

void main()
{
	ivec2 screenPosition;
	screenPosition.x = ActiveRect.x + int(position.x * float(ActiveRect.z) + 0.5f);
	screenPosition.y = ActiveRect.y + int(position.y * float(ActiveRect.w) + 0.5f);

	LocalOffset.x = screenPosition.x;
	LocalOffset.y = screenPosition.y;
	LocalOffset.z = (screenPosition.y - WaterLevel) / float(GameResolution.y);

	gl_Position.x = float(screenPosition.x) / float(GameResolution.x) * 2.0f - 1.0f;
	gl_Position.y = float(screenPosition.y) / float(GameResolution.y) * 2.0f - 1.0f;
	gl_Position.z = 0.5f;	// Has an effect only if depth write is enabled (it is for the priority plane, i.e. when PriorityFlag == 1)
	gl_Position.w = 1.0f;
}



## ----- Fragment -----------------------------------------------------------------

in vec3 LocalOffset;
out vec4 FragColor;

uniform isamplerBuffer IndexTexture;
uniform isamplerBuffer PatternCacheTexture;
uniform sampler2D PaletteTexture;
uniform ivec4 PlayfieldSize;
uniform int PriorityFlag;		// 0 or 1

#ifdef HORIZONTAL_SCROLLING
	uniform isamplerBuffer HScrollOffsetsTexture;
#else
	uniform int ScrollOffsetX;
#endif

#ifdef VERTICAL_SCROLLING
	uniform isamplerBuffer VScrollOffsetsTexture;
	uniform int VScrollOffsetBias;
#else
	uniform int ScrollOffsetY;
#endif

void main()
{
	int ix = int(LocalOffset.x);
	int iy = int(LocalOffset.y);

#ifdef HORIZONTAL_SCROLLING
	int scrollOffsetLookupX = texelFetch(HScrollOffsetsTexture, iy).x;
#else
	int scrollOffsetLookupX = ScrollOffsetX;
#endif

#ifdef VERTICAL_SCROLLING
	int scrollOffsetLookupY = texelFetch(VScrollOffsetsTexture, ((ix - VScrollOffsetBias) & 0x1f0) >> 4).x;
#else
	int scrollOffsetLookupY = ScrollOffsetY;
#endif

	ix = (ix + scrollOffsetLookupX);
	iy = (iy + scrollOffsetLookupY) & (PlayfieldSize.y - 1);

#ifdef NOREPEAT_SCROLLOFFSETS
	if (ix < 0 || ix >= PlayfieldSize.x)
		discard;
#else
	ix &= (PlayfieldSize.x - 1);
#endif

	int patternX = ix / 8;
	int patternY = iy / 8;
	int localX = ix & 0x07;
	int localY = iy & 0x07;

	int patternIndex = texelFetch(IndexTexture, patternX + patternY * PlayfieldSize.z).x;
	if ((patternIndex >> 15) != PriorityFlag)
		discard;

	int atex = (patternIndex >> 9) & 0x30;
	int flipVariation = (patternIndex >> 11) & 0x03;
	int patternCacheLookupIndex = localX + localY * 8 + flipVariation * 64 + (patternIndex & 0x07ff) * 256;
	int paletteIndex = atex + texelFetch(PatternCacheTexture, patternCacheLookupIndex).x;

	vec4 color = texture(PaletteTexture, vec2(float(paletteIndex) / 256.0f, LocalOffset.z + 0.5f));
#ifdef ALPHA_TEST
	if (color.a < 0.01f)
		discard;
#endif

	FragColor = color;
}



## ----- TECH ---------------------------------------------------------------------

technique Standard
{
	blendfunc = opaque;
	vs = Shared + Vertex;
	fs = Shared + Fragment;
	vertexattrib[0] = position;
}

technique HorizontalScrolling : Standard
{
	define = HORIZONTAL_SCROLLING;
}

technique HorizontalScrollingNoRepeat : Standard
{
	define = HORIZONTAL_SCROLLING;
	define = NOREPEAT_SCROLLOFFSETS;
}

technique VerticalScrolling : Standard
{
	define = VERTICAL_SCROLLING;
}

technique HorizontalVerticalScrolling : Standard
{
	define = HORIZONTAL_SCROLLING;
	define = VERTICAL_SCROLLING;
}

technique Standard_AlphaTest : Standard
{
	define = ALPHA_TEST;
}

technique HorizontalScrolling_AlphaTest : Standard
{
	define = HORIZONTAL_SCROLLING;
	define = ALPHA_TEST;
}

technique HorizontalScrollingNoRepeat_AlphaTest : Standard
{
	define = HORIZONTAL_SCROLLING;
	define = NOREPEAT_SCROLLOFFSETS;
	define = ALPHA_TEST;
}

technique VerticalScrolling_AlphaTest : Standard
{
	define = VERTICAL_SCROLLING;
	define = ALPHA_TEST;
}

technique HorizontalVerticalScrolling_AlphaTest : Standard
{
	define = HORIZONTAL_SCROLLING;
	define = VERTICAL_SCROLLING;
	define = ALPHA_TEST;
}
